/******************************************************************************
 *
 *		M M A 8 4 5 1 Q    A C C E L E R O M E T E R   C O D E  
 *
 */
#include "accel.h"
#include "display.h"
#include <math.h>
#include "p18lf14k22.h"

#define BATT_LEVEL_MV		2800			// 2.8 V
#define CAL_F_ADDRESS 		0
#define CAL_R_ADDRESS       4
#define CAL_D_ADDRESS       8
#define FLAG_ADDRESS		12

unsigned char rawdata[6];

float afraw, arraw;
float gf, gr;
float amax;
int signmax, axismax;
int peakcount;
unsigned int battlevel;

void accInitialise(void)
{
	dispInitialise();
	// Set up I2C
	i2cInitialise();
	// Set up pins
	CAL_TRIS = 1;
	INT1_TRIS = 1;
	INT2_TRIS = 1;
	INTCON2bits.INTEDG1 = 0;				// Int 1 falling edge
	INTCON3bits.INT1IP = 0;					// Int 1 low prio
	INTCON2bits.INTEDG2 = 0;				// Int 2 falling edge
	INTCON3bits.INT2IP = 0;					// Int 2 low prio
	CAL_PULLUP = 1;
	INTCON2bits.RABPU = 0;					// enable pullups
	INTCON2bits.RABIP = 0;					// Cal button low priority
	CAL_IOC = 1;							// Enable interrupt on change
	i2cSetByte(ACC_ADDR, CTRL_REG1, i2cGetByte(ACC_ADDR, CTRL_REG1) & 0xfe);
	// Set up Battery Monitor
	ADCON0bits.CHS = 0xf;                                   // convert FVR
	ADCON1bits.PVCFG = 0;					// +ve ref Vdd
	ADCON1bits.NVCFG = 0;					// -ve ref Vss
	ADCON2bits.ADFM	= 1;					// Result right justified
	ADCON2bits.ACQT = 0x4;					// 8 Tad acquisition time
	ADCON2bits.ADCS = 0x5;					// Tad = Fosc/16 = 1MHz
	VREFCON0bits.FVR1S0 = 1;				// FVR = 1.024V
	VREFCON0bits.FVR1EN = 1;				// FVR on
	while(VREFCON0bits.FVR1ST == 0);		// wait till FVR is stable
	ADCON0bits.ADON = 1;					// turn ADC on
	battlevel = (unsigned int)(1047552 / BATT_LEVEL_MV);
}		

void accStart(void)
{
	uchar temp;
	// Put into active mode
	i2cSetByte(ACC_ADDR, CTRL_REG1, i2cGetByte(ACC_ADDR, CTRL_REG1) | 0x01);
    amax = 0.0;
    signmax = 0;
    axismax = 0;
    peakcount = 0;
	// get calibration value
	gf = 0.0;
	gr = 0.0;

	if(eeReadChar(FLAG_ADDRESS) == 0x00) {	// Good cal value
		gf = eeReadFloat(CAL_F_ADDRESS);
		gr = eeReadFloat(CAL_R_ADDRESS);
	}	
	else {
		eeWriteFloat(CAL_F_ADDRESS, 0.00);
		eeWriteFloat(CAL_R_ADDRESS, 0.00);
		eeWriteChar(FLAG_ADDRESS, 0x00);
	}

	// Enable interrupts
	INTCON3bits.INT1IF = 0;					// Clear int1 flag
	INTCON3bits.INT1IE = 1;					// enable int1
	temp = CAL_PIN;							// clear any mismatch and 
	INTCONbits.RABIF = 0;					// Clear IOC flag
	INTCONbits.RABIE = 1;					// Enable interrupt on change 
}	
	
void accSet(uchar reg, uchar value)
{
	i2cSetByte(ACC_ADDR, reg, value);	
}

uchar accGet(uchar reg)
{
	return i2cGetByte(ACC_ADDR, reg);
}


#define HOLD_TIME    32   // 32 units of 160 ms = 5.12 sec
void accInt1isr(void)
{
	int accx, accy, accz;
	float temp;
	static int sign, axis;
	static float af,ar;
	
	peakcount++;
    // Calculate acceleration in G
	i2cRead(ACC_ADDR, OUTPUT_REGS, 6, rawdata);
	accx = ((unsigned int)rawdata[0] << 8) + rawdata[1];
	accy = ((unsigned int)rawdata[2] << 8) + rawdata[3];
	accz = ((unsigned int)rawdata[4] << 8) + rawdata[5];
	afraw = -(float)accz / 16384.0;
	arraw = (float)accx / 16384.0;
	
	// Offset 
	af = afraw - gf;
	ar = arraw - gr;
 		
	// Find largest
	if(fabs(af) >= fabs(ar)){
    	sign = af > 0.0 ? 0 : 1;
    	temp = fabs(af);
    	axis = FWD_REV;
    }  
   	else {
    	sign = ar > 0.0 ? 0 : 1;
    	temp = fabs(ar);
    	axis = LFT_RGT;
    } 
   
	// Peak hold
	if((temp >= amax) || (peakcount >= HOLD_TIME)){
    	amax = temp;
    	signmax = sign;
    	axismax = axis;
    	peakcount = 0;
    }
    if(signmax == 1)dispDec2(-amax, axismax);
    else dispDec2(amax, axismax);
    
    ADCON0bits.GO = 1;
	while(ADCON0bits.GO);
	if(ADRES > battlevel) dispBatt();
}
	

void accCalisr()
{
    if(CAL_PIN == 0) {
        gf = afraw;
        gr = arraw;
       	eeWriteFloat(CAL_F_ADDRESS, gf);
		eeWriteFloat(CAL_R_ADDRESS, gr);
		peakcount = HOLD_TIME;
	}	
}	
	
